<?php
// MT5 Authentication and User Information Fetcher
error_reporting(E_ALL);
ini_set('display_errors', 1);

class CMT5Request {
    private $m_curl = null;
    private $m_server = "";
    private $m_authenticated = false;
    public $last_error = "";
    public $last_response = "";
    public $last_http_code = 0;

    public function Init($server) {
        $this->Shutdown();
        if($server == null) {
            $this->last_error = "Server address cannot be null";
            return false;
        }
        
        $this->m_curl = curl_init();
        if($this->m_curl == null) {
            $this->last_error = "Failed to initialize cURL";
            return false;
        }
        
        // cURL options
        curl_setopt_array($this->m_curl, [
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_MAXCONNECTS => 1,
            CURLOPT_HTTPHEADER => ['Connection: Keep-Alive'],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HEADER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_USERAGENT => 'MT5-WebAPI-Client/1.0',
            CURLOPT_COOKIEFILE => '', // Enable cookie handling
            CURLOPT_COOKIEJAR => '', // Enable cookie handling
        ]);
        
        $this->m_server = $server;
        $this->last_error = "";
        return true;
    }

    public function Shutdown() {
        if($this->m_curl != null) {
            curl_close($this->m_curl);
        }
        $this->m_curl = null;
        $this->m_authenticated = false;
    }

    private function executeRequest($path, $is_post = false, $post_data = null) {
        if($this->m_curl == null) {
            $this->last_error = "cURL not initialized";
            return false;
        }
        
        curl_setopt($this->m_curl, CURLOPT_POST, $is_post);
        curl_setopt($this->m_curl, CURLOPT_URL, 'https://' . $this->m_server . $path);
        
        if($is_post && $post_data !== null) {
            curl_setopt($this->m_curl, CURLOPT_POSTFIELDS, $post_data);
        }
        
        $response = curl_exec($this->m_curl);
        $this->last_http_code = curl_getinfo($this->m_curl, CURLINFO_HTTP_CODE);
        
        if($response === false) {
            $this->last_error = 'CURL error: ' . curl_error($this->m_curl);
            return false;
        }
        
        // Split headers and body
        $header_size = curl_getinfo($this->m_curl, CURLINFO_HEADER_SIZE);
        $headers = substr($response, 0, $header_size);
        $body = substr($response, $header_size);
        
        $this->last_response = $body;
        
        return $body;
    }

    public function Get($path) {
        return $this->executeRequest($path, false);
    }

    public function Post($path, $body) {
        return $this->executeRequest($path, true, $body);
    }

    public function Auth($login, $password, $build, $agent) {
        if($this->m_curl == null) {
            $this->last_error = "cURL not initialized";
            return false;
        }
        
        // Step 1: Auth start
        $path = '/api/auth/start?version=' . $build . '&agent=' . urlencode($agent) . 
                '&login=' . $login . '&type=manager';
        
        $result = $this->Get($path);
        if($result === false) {
            $this->last_error = "Auth start failed: " . $this->last_error . " (HTTP: " . $this->last_http_code . ")";
            return false;
        }
        
        $auth_start_answer = json_decode($result);
        if(!$auth_start_answer) {
            $this->last_error = "Invalid JSON response from auth start: " . $result;
            return false;
        }
        
        if(!isset($auth_start_answer->retcode) || (int)$auth_start_answer->retcode != 0) {
            $this->last_error = "Auth start error: " . ($auth_start_answer->retcode ?? 'Unknown');
            if(isset($auth_start_answer->description)) {
                $this->last_error .= " - " . $auth_start_answer->description;
            }
            return false;
        }
        
        // Step 2: Process challenge
        if(!isset($auth_start_answer->srv_rand)) {
            $this->last_error = "No srv_rand in auth response: " . $result;
            return false;
        }
        
        // Convert password to UTF-16LE and create first MD5 hash
        $password_utf16 = mb_convert_encoding($password, 'UTF-16LE', 'UTF-8');
        $password_md5 = md5($password_utf16, true);
        
        // Append "WebAPI" as bytes (not as string)
        $webapi_bytes = 'WebAPI'; // This will be treated as bytes in binary context
        $password_hash_input = $password_md5 . $webapi_bytes;
        
        // Create final password hash
        $password_hash = md5($password_hash_input, true);
        
        // Convert hex srv_rand to binary
        $srv_rand_bin = hex2bin($auth_start_answer->srv_rand);
        if($srv_rand_bin === false) {
            $this->last_error = "Failed to convert srv_rand from hex: " . $auth_start_answer->srv_rand;
            return false;
        }
        
        // Calculate srv_rand_answer = MD5(password_hash + srv_rand_bin)
        $srv_rand_answer_input = $password_hash . $srv_rand_bin;
        $srv_rand_answer = md5($srv_rand_answer_input);
        
        // Generate client random
        $cli_rand_buf = random_bytes(16);
        $cli_rand = bin2hex($cli_rand_buf);
        
        // Step 3: Auth answer
        $path = '/api/auth/answer?srv_rand_answer=' . $srv_rand_answer . '&cli_rand=' . $cli_rand;
        $result = $this->Get($path);
        if($result === false) {
            $this->last_error = "Auth answer failed: " . $this->last_error . " (HTTP: " . $this->last_http_code . ")";
            return false;
        }
        
        $auth_answer_answer = json_decode($result);
        if(!$auth_answer_answer) {
            $this->last_error = "Invalid JSON response from auth answer: " . $result;
            return false;
        }
        
        if(!isset($auth_answer_answer->retcode) || (int)$auth_answer_answer->retcode != 0) {
            $this->last_error = "Auth answer error: " . ($auth_answer_answer->retcode ?? 'Unknown');
            if(isset($auth_answer_answer->description)) {
                $this->last_error .= " - " . $auth_answer_answer->description;
            }
            return false;
        }
        
        // Step 4: Verify server response
        if(!isset($auth_answer_answer->cli_rand_answer)) {
            $this->last_error = "No cli_rand_answer in auth response: " . $result;
            return false;
        }
        
        // Calculate expected cli_rand_answer = MD5(password_hash + cli_rand_buf)
        $cli_rand_answer_input = $password_hash . $cli_rand_buf;
        $expected_cli_rand_answer = md5($cli_rand_answer_input);
        
        if($expected_cli_rand_answer != $auth_answer_answer->cli_rand_answer) {
            $this->last_error = "Auth answer error: invalid client answer. Expected: " . 
                               $expected_cli_rand_answer . " Got: " . $auth_answer_answer->cli_rand_answer;
            return false;
        }
        
        $this->m_authenticated = true;
        return true;
    }
    
    public function isAuthenticated() {
        return $this->m_authenticated;
    }
}

function connectToMT5() {
    $login = 1023;
    $password = "Ab@it1023sky#";
    $build = 4330;
    $agent = "WebAPI";
    $server = "mt5.skylinkstrader.com:443";
    
    $request = new CMT5Request();
    
    // Initialize
    if(!$request->Init($server)) {
        return ["error" => "Init failed: " . $request->last_error];
    }
    
    // Authenticate
    if(!$request->Auth($login, $password, $build, $agent)) {
        return ["error" => "Auth failed: " . $request->last_error, 
                "http_code" => $request->last_http_code];
    }
    
    return [
        "success" => true,
        "request" => $request
    ];
}

function fetchUserDetails($request, $userId) {
    // Fetch user details
    $result = $request->Get('/api/user/get?login=' . $userId);
    if($result === false) {
        return ["error" => "Get user failed: " . $request->last_error,
                "http_code" => $request->last_http_code];
    }
    
    $userData = json_decode($result, true);
    if(!$userData) {
        return ["error" => "Invalid JSON response: " . $result];
    }
    
    // Check if the response has the expected structure
    if(isset($userData['retcode'])) {
        // If retcode is "0 Done", it means success but we need to check the answer field
        if($userData['retcode'] === '0 Done' && isset($userData['answer'])) {
            return $userData;
        }
        // If retcode is just "0", it's also success
        elseif($userData['retcode'] === '0' && isset($userData['answer'])) {
            return $userData;
        }
        // Handle other retcode values as errors
        elseif($userData['retcode'] !== '0' && $userData['retcode'] !== '0 Done') {
            return ["error" => "API Error: " . $userData['retcode'] . 
                    (isset($userData['description']) ? " - " . $userData['description'] : "")];
        }
    }
    
    // If we get here, the response structure is unexpected
    return ["error" => "Unexpected API response format: " . $result];
}

function fetchPositions($request, $login, $perPage = 100) {
    // Fetch positions page-by-page using /api/position/get_page
    $per = max(1, min(100, intval($perPage)));
    $offset = 0;
    $all = [];

    while (true) {
        $path = '/api/position/get_page?login=' . urlencode($login) . '&offset=' . $offset . '&total=' . $per;
        $result = $request->Get($path);
        if ($result === false) {
            return ["error" => "Get positions failed: " . $request->last_error, "http_code" => $request->last_http_code];
        }

        $data = json_decode($result, true);
        if (!$data) {
            return ["error" => "Invalid JSON response: " . $result];
        }

        if (isset($data['retcode']) && (strval($data['retcode']) !== '0' && strval($data['retcode']) !== '0 Done')) {
            return ["error" => "API Error: " . $data['retcode'] . (isset($data['description']) ? " - " . $data['description'] : "")];
        }

        $answer = isset($data['answer']) && is_array($data['answer']) ? $data['answer'] : [];
        $count = count($answer);
        if ($count > 0) {
            foreach ($answer as $item) {
                $all[] = $item;
            }
        }

        // If fewer than requested items returned, we're done
        if ($count < $per) {
            break;
        }

        $offset += $count;
        // safety break to avoid infinite loops
        if ($offset > 1000000) break;
    }

    return ["positions" => $all];
}

// Dev by Abbas: try to fetch all open positions server-wide (tries common endpoints)
function fetchAllPositions($request, $perPage = 200) {
    $try_paths = [
        '/api/position/get_batch',
        '/positions',
        '/api/position/get_page?offset=0&total=' . intval(max(1, min(100, $perPage)))
    ];

    foreach ($try_paths as $path) {
        $result = $request->Get($path);
        if ($result === false) continue;
        $data = json_decode($result, true);
        if (!$data) continue;

        if (isset($data['answer']) && is_array($data['answer']) && count($data['answer']) > 0) {
            return ["positions" => $data['answer']];
        }
        if (isset($data['positions']) && is_array($data['positions']) && count($data['positions']) > 0) {
            return ["positions" => $data['positions']];
        }
        if (is_array($data) && array_keys($data) === range(0, count($data) - 1) && count($data) > 0) {
            return ["positions" => $data];
        }
    }
    // fallback: try discovering logins and fetch per-login
    $bylog = fetchAllPositionsByLogins($request, min(100, $perPage), 200);
    if (isset($bylog['positions'])) return $bylog;
    return ["error" => "No server-wide positions endpoint succeeded and per-login scan failed: " . ($bylog['error'] ?? 'unknown')];
}

// Dev by Abbas: try to discover all client logins via common list endpoints
function discoverAllLogins($request, $max = 500) {
    $candidates = [
        '/api/user/list', '/api/client/list', '/api/user/get_list', '/api/client/get_list',
        '/api/users', '/api/clients', '/api/account/list', '/api/account/get_list'
    ];

    $logins = [];
    foreach ($candidates as $path) {
        $res = $request->Get($path);
        if ($res === false) continue;
        $data = json_decode($res, true);
        if (!$data) continue;

        // normalize answer/rows/top-level array
        if (isset($data['answer']) && is_array($data['answer'])) $rows = $data['answer'];
        elseif (isset($data['rows']) && is_array($data['rows'])) $rows = $data['rows'];
        elseif (is_array($data) && array_keys($data) === range(0, count($data) - 1)) $rows = $data;
        else $rows = [];

        foreach ($rows as $r) {
            if (!is_array($r)) continue;
            if (isset($r['Login'])) $logins[] = intval($r['Login']);
            elseif (isset($r['login'])) $logins[] = intval($r['login']);
            elseif (isset($r['LoginID'])) $logins[] = intval($r['LoginID']);
            elseif (isset($r['id'])) $logins[] = intval($r['id']);
        }

        if (count($logins) >= $max) break;
    }

    $logins = array_values(array_unique(array_filter($logins, function($v){ return $v > 0; })));
    if (count($logins) > $max) $logins = array_slice($logins, 0, $max);
    return $logins;
}

// Dev by Abbas: when no server-wide positions endpoint exists, try per-login fetch
function fetchAllPositionsByLogins($request, $perLogin = 100, $maxLogins = 200) {
    $logins = discoverAllLogins($request, $maxLogins);
    if (empty($logins)) return ["error" => "No client list endpoints returned logins."];

    $all = [];
    foreach ($logins as $login) {
        $res = fetchPositions($request, $login, $perLogin);
        if (isset($res['error'])) {
            // skip failing logins but continue
            continue;
        }
        if (!empty($res['positions']) && is_array($res['positions'])) {
            foreach ($res['positions'] as $p) $all[] = $p;
        }
    }

    if (empty($all)) return ["error" => "No positions found when scanning discovered logins."];
    return ["positions" => $all];
}

// Dev by Abbas: aggregate open positions by symbol into summary rows
function aggregatePositionsBySymbol(array $positions) {
    $summary = [];
    foreach ($positions as $pos) {
        $sym = $pos['Symbol'] ?? $pos['symbol'] ?? '';
        if ($sym === '') continue;

        $vol = floatval($pos['Volume'] ?? $pos['volume'] ?? 0.0);
        $open = floatval($pos['OpenPrice'] ?? $pos['open_price'] ?? $pos['openprice'] ?? 0.0);
        $price = floatval($pos['Price'] ?? $pos['price'] ?? 0.0);
        $typeRaw = strtolower((string)($pos['Type'] ?? $pos['type'] ?? ''));
        $type = (strpos($typeRaw, 'buy') !== false || $typeRaw === 'b') ? 'buy' : 'sell';
        $profit = floatval($pos['Profit'] ?? $pos['profit'] ?? 0.0);
        $swap = floatval($pos['Swap'] ?? $pos['swap'] ?? 0.0);
        $commission = floatval($pos['Commission'] ?? $pos['commission'] ?? 0.0);

        if (!isset($summary[$sym])) {
            $summary[$sym] = [
                'symbol' => $sym,
                'positions_count' => 0,
                'buy_volume' => 0.0,
                'sell_volume' => 0.0,
                'sum_open_px_vol' => 0.0,
                'sum_price_vol' => 0.0,
                'total_volume' => 0.0,
                'total_profit' => 0.0,
                'total_swap' => 0.0,
                'total_commission' => 0.0,
            ];
        }

        $summary[$sym]['positions_count'] += 1;
        $summary[$sym]['total_volume'] += $vol;
        $summary[$sym]['sum_open_px_vol'] += $open * $vol;
        $summary[$sym]['sum_price_vol'] += ($price > 0 ? $price : $open) * $vol;
        $summary[$sym]['total_profit'] += $profit;
        $summary[$sym]['total_swap'] += $swap;
        $summary[$sym]['total_commission'] += $commission;

        if ($type === 'buy') {
            $summary[$sym]['buy_volume'] += $vol;
        } else {
            $summary[$sym]['sell_volume'] += $vol;
        }
    }

    $result = [];
    foreach ($summary as $sym => $row) {
        $total_vol = $row['total_volume'] > 0 ? $row['total_volume'] : 0.0;
        $avg_open = $total_vol > 0 ? $row['sum_open_px_vol'] / $total_vol : 0.0;
        $avg_price = $total_vol > 0 ? $row['sum_price_vol'] / $total_vol : 0.0;
        $net_volume = $row['buy_volume'] - $row['sell_volume'];
        $exposure = $row['sum_price_vol'];

        $result[] = [
            'symbol' => $sym,
            'positions_count' => $row['positions_count'],
            'buy_volume' => $row['buy_volume'],
            'sell_volume' => $row['sell_volume'],
            'net_volume' => $net_volume,
            'avg_open_price' => $avg_open,
            'avg_price' => $avg_price,
            'total_profit' => $row['total_profit'],
            'total_swap' => $row['total_swap'],
            'total_commission' => $row['total_commission'],
            'exposure' => $exposure,
        ];
    }

    usort($result, function($a, $b){
        return abs($b['net_volume']) <=> abs($a['net_volume']);
    });

    return $result;
}

// Handle form submission
$connectionResult = null;
$userDetails = null;
$debugInfo = [];

if($_SERVER['REQUEST_METHOD'] === 'POST') {
    if(isset($_POST['connect']) || isset($_POST['fetch_all']) || isset($_POST['scan_list'])) {
        $connectionResult = connectToMT5();

        if(isset($connectionResult['success']) && $connectionResult['success'] === true) {
            // If Scan Logins pressed -> use provided login list/ranges to fetch positions
            if (isset($_POST['scan_list'])) {
                $raw = trim($_POST['login_list_text'] ?? '');
                $logins = [];
                if ($raw !== '') {
                    // split by comma, whitespace or newline
                    $parts = preg_split('/[\s,;]+/', $raw);
                    foreach ($parts as $p) {
                        $p = trim($p);
                        if ($p === '') continue;
                        // range e.g. 1000-1010
                        if (preg_match('/^(\d+)\s*-\s*(\d+)$/', $p, $m)) {
                            $a = intval($m[1]); $b = intval($m[2]);
                            if ($a > $b) { $t = $a; $a = $b; $b = $t; }
                            for ($i = $a; $i <= $b; $i++) {
                                $logins[] = $i;
                            }
                        } elseif (preg_match('/^\d+$/', $p)) {
                            $logins[] = intval($p);
                        }
                    }
                    $logins = array_values(array_unique($logins));
                }

                if (empty($logins)) {
                    $debugInfo['positions_error'] = ['error' => 'No valid logins supplied to scan.'];
                } else {
                    // limit to 1000 logins to avoid runaway scans
                    $logins = array_slice($logins, 0, 1000);
                    $all = [];
                    $pos_errors = [];
                    foreach ($logins as $lg) {
                        $res = fetchPositions($connectionResult['request'], $lg, 100);
                        if (isset($res['error'])) {
                            $pos_errors[$lg] = $res['error'];
                            continue;
                        }
                        if (!empty($res['positions']) && is_array($res['positions'])) {
                            foreach ($res['positions'] as $p) $all[] = $p;
                        }
                    }
                    if (empty($all)) {
                        $debugInfo['positions_error'] = ['error' => 'No positions found scanning supplied logins.', 'per_login_errors' => $pos_errors];
                    } else {
                        $debugInfo['positions'] = $all;
                        if (!empty($pos_errors)) $debugInfo['positions_partial_errors'] = $pos_errors;
                    }
                }
            }
            // If Fetch All button pressed -> server-wide positions
            elseif (isset($_POST['fetch_all'])) {
                $positionsResult = fetchAllPositions($connectionResult['request'], 200);
                if (isset($positionsResult['error'])) {
                    $debugInfo['positions_error'] = $positionsResult;
                } else {
                    $debugInfo['positions'] = $positionsResult['positions'];
                }
            }
            // Else regular Connect: only fetch when user_id provided
            elseif (isset($_POST['user_id']) && !empty($_POST['user_id'])) {
                $userId = intval($_POST['user_id']);
                $userDetails = fetchUserDetails($connectionResult['request'], $userId);
                // Fetch positions for this user as well
                $positionsResult = fetchPositions($connectionResult['request'], $userId, 100);
                if (isset($positionsResult['error'])) {
                    $debugInfo['positions_error'] = $positionsResult;
                } else {
                    $debugInfo['positions'] = $positionsResult['positions'];
                }
            }

            // Debug: Store the raw response for analysis (if available)
            $debugInfo['user_response'] = $connectionResult['request']->last_response;
            $debugInfo['user_http_code'] = $connectionResult['request']->last_http_code;
        }

        // Shutdown the request
        if(isset($connectionResult['request'])) {
            $connectionResult['request']->Shutdown();
        }
    }
}

// Dev by Abbas: prepare symbol summary for rendering at top
$symbolSummary = [];
if (!empty($debugInfo['positions']) && is_array($debugInfo['positions'])) {
    $symbolSummary = aggregatePositionsBySymbol($debugInfo['positions']);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>MT5 Connection Interface</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; line-height: 1.6; background-color: #f5f5f5; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        h1 { color: #333; text-align: center; }
        .form-container { margin-bottom: 20px; }
        .form-group { margin-bottom: 15px; }
        label { display: block; margin-bottom: 5px; font-weight: bold; color: #555; }
        input[type="number"] { padding: 10px; width: 100%; border: 1px solid #ddd; border-radius: 4px; box-sizing: border-box; }
        button { background: #4CAF50; color: white; padding: 12px 20px; border: none; border-radius: 4px; cursor: pointer; width: 100%; font-size: 16px; }
        button:hover { background: #45a049; }
        .result-container { margin-top: 20px; padding: 15px; border-radius: 4px; }
        .success { background-color: #dff0d8; color: #3c763d; border: 1px solid #d6e9c6; }
        .error { background-color: #f2dede; color: #a94442; border: 1px solid #ebccd1; }
        .user-table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        .user-table th, .user-table td { border: 1px solid #ddd; padding: 10px; text-align: left; }
        .user-table th { background-color: #f2f2f2; }
        .info-box { background-color: #e7f3fe; border-left: 4px solid #2196F3; padding: 15px; margin-bottom: 20px; }
        .debug { background: #333; color: #fff; padding: 15px; border-radius: 5px; margin-top: 20px; font-family: monospace; white-space: pre-wrap; }
    </style>
</head>
<body>
    <div class="container">
        <h1>MT5 Connection Interface</h1>
        
        <div class="info-box">
            <p><strong>Manager Account:</strong> 1023</p>
            <p>Click the "Connect to MT5" button to establish a connection with the MT5 server using your manager account.</p>
        </div>
        <!-- Dev by Abbas: Positions Summary by Symbol (top) -->
        <?php if (!empty($symbolSummary)): ?>
            <div class="result-container">
                <h2>Positions Summary by Symbol</h2>
                <table class="user-table">
                    <thead>
                        <tr>
                            <th>Symbol</th>
                            <th>Positions</th>
                            <th>Buy Vol</th>
                            <th>Sell Vol</th>
                            <th>Net Vol</th>
                            <th>Avg Open</th>
                            <th>Avg Price</th>
                            <th>Total Profit</th>
                            <th>Total Swap</th>
                            <th>Total Commission</th>
                            <th>Exposure</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($symbolSummary as $s): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($s['symbol']); ?></td>
                                <td><?php echo intval($s['positions_count']); ?></td>
                                <td><?php echo number_format($s['buy_volume'], 2); ?></td>
                                <td><?php echo number_format($s['sell_volume'], 2); ?></td>
                                <td><?php echo number_format($s['net_volume'], 2); ?></td>
                                <td><?php echo number_format($s['avg_open_price'], 5); ?></td>
                                <td><?php echo number_format($s['avg_price'], 5); ?></td>
                                <td><?php echo number_format($s['total_profit'], 2); ?></td>
                                <td><?php echo number_format($s['total_swap'], 2); ?></td>
                                <td><?php echo number_format($s['total_commission'], 2); ?></td>
                                <td><?php echo number_format($s['exposure'], 2); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
        
        <div class="form-container">
            <form method="POST">
                <div class="form-group">
                    <label for="user_id">Trading Account ID (optional):</label>
                    <input type="number" id="user_id" name="user_id" min="1" 
                           value="<?php echo isset($_POST['user_id']) ? htmlspecialchars($_POST['user_id']) : ''; ?>"
                           placeholder="Enter trading account ID to fetch details">
                </div>
                <div class="form-group">
                    <label for="login_list_text">Or paste login IDs / ranges (one per line or comma separated):</label>
                    <textarea id="login_list_text" name="login_list_text" rows="4" style="width:100%;box-sizing:border-box;"><?php echo isset($_POST['login_list_text']) ? htmlspecialchars($_POST['login_list_text']) : ''; ?></textarea>
                </div>
                <div style="display:flex;gap:10px;">
                    <button type="submit" name="connect" value="1" style="flex:1;">Connect to MT5</button>
                    <button type="submit" name="fetch_all" value="1" style="flex:1;background:#2196F3;">Fetch All Positions</button>
                    <button type="submit" name="scan_list" value="1" style="flex:1;background:#FF9800;">Scan Logins</button>
                </div>
            </form>
        </div>
        
        <?php if ($connectionResult): ?>
            <div class="result-container <?php echo isset($connectionResult['success']) ? 'success' : 'error'; ?>">
                <h2>Connection Result</h2>
                
                <?php if (isset($connectionResult['success']) && $connectionResult['success'] === true): ?>
                    <p><strong>Success:</strong> Connected to MT5 server successfully!</p>
                <?php else: ?>
                    <p><strong>Error:</strong> <?php echo htmlspecialchars($connectionResult['error']); ?></p>
                    <?php if (isset($connectionResult['http_code'])): ?>
                        <p><strong>HTTP Code:</strong> <?php echo htmlspecialchars($connectionResult['http_code']); ?></p>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        <?php endif; ?>
        
        <!-- Open Positions Section -->
        <?php if (!empty($debugInfo['positions'])): ?>
            <div class="result-container">
                <h2>Open Positions (<?php echo count($debugInfo['positions']); ?>)</h2>
                <table class="user-table">
                    <thead>
                        <tr>
                            <th>Position</th>
                            <th>Symbol</th>
                            <th>Login</th>
                            <th>Type</th>
                            <th>Volume</th>
                            <th>Open Price</th>
                            <th>Price</th>
                            <th>Profit</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($debugInfo['positions'] as $pos): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($pos['Position'] ?? $pos['Ticket'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($pos['Symbol'] ?? $pos['symbol'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($pos['Login'] ?? $pos['login'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($pos['Type'] ?? $pos['type'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($pos['Volume'] ?? $pos['volume'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($pos['OpenPrice'] ?? $pos['open_price'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($pos['Price'] ?? $pos['price'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($pos['Profit'] ?? $pos['profit'] ?? ''); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php elseif (!empty($debugInfo['positions_error'])): ?>
            <div class="result-container error">
                <h2>Positions Error</h2>
                <p><?php echo htmlspecialchars(json_encode($debugInfo['positions_error'])); ?></p>
            </div>
        <?php endif; ?>
        
        <?php if ($userDetails): ?>
            <div class="result-container">
                <h2>User Details</h2>
                
                <?php if (isset($userDetails['error'])): ?>
                    <div class="error">
                        <strong>Error:</strong> <?php echo htmlspecialchars($userDetails['error']); ?>
                        <?php if (isset($userDetails['http_code'])): ?>
                            <br><strong>HTTP Code:</strong> <?php echo htmlspecialchars($userDetails['http_code']); ?>
                        <?php endif; ?>
                    </div>
                    
                    <?php if (!empty($debugInfo)): ?>
                        <div class="debug">
                            <h3>Debug Information:</h3>
                            <pre><?php echo htmlspecialchars(print_r($debugInfo, true)); ?></pre>
                        </div>
                    <?php endif; ?>
                    
                <?php elseif (isset($userDetails['answer'])): ?>
                    <div class="success">
                        Successfully retrieved user details for ID: <?php echo htmlspecialchars($_POST['user_id']); ?>
                    </div>
                    
                    <table class="user-table">
                        <thead>
                            <tr>
                                <th>Property</th>
                                <th>Value</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($userDetails['answer'] as $key => $value): ?>
                                <tr>
                                    <td><strong><?php echo htmlspecialchars($key); ?></strong></td>
                                    <td>
                                        <?php 
                                        if (is_array($value)) {
                                            echo htmlspecialchars(json_encode($value, JSON_PRETTY_PRINT));
                                        } else if (is_bool($value)) {
                                            echo $value ? 'true' : 'false';
                                        } else {
                                            echo htmlspecialchars($value);
                                        }
                                        ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <div class="error">
                        Unexpected response format from MT5 API.
                        <pre><?php echo htmlspecialchars(print_r($userDetails, true)); ?></pre>
                        
                        <?php if (!empty($debugInfo)): ?>
                            <div class="debug">
                                <h3>Raw Response:</h3>
                                <pre><?php echo htmlspecialchars($debugInfo['user_response']); ?></pre>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>