<?php
// cli/monitor_triggers.php - Monitor MT5 events and send WhatsApp notifications based on saved rules
// Usage: php cli/monitor_triggers.php

require_once __DIR__ . '/../config/bootstrap.php';
require_once __DIR__ . '/../src/MT5WebAPI.php';
require_once __DIR__ . '/../src/WhatsAppClient.php';

$pdo = get_pdo();
$mt5 = new MT5WebAPI();
$wa = new WhatsAppClient();

echo "Starting trigger monitor...\n";

function get_enabled_notifications(PDO $pdo): array {
    $stmt = $pdo->query("SELECT * FROM notifications WHERE enabled = 1 ORDER BY id ASC");
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    foreach ($rows as &$r) {
        $r['account_logins'] = json_decode($r['account_logins'], true) ?: [];
    }
    return $rows;
}

function get_checkpoint(PDO $pdo, int $notificationId): array {
    $stmt = $pdo->prepare("SELECT * FROM notification_checkpoints WHERE notification_id = ?");
    $stmt->execute([$notificationId]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) {
        $now = date('Y-m-d H:i:s', time() - 60); // start 1 min ago
        $pdo->prepare("INSERT INTO notification_checkpoints (notification_id, last_checked_at) VALUES (?, ?)")
            ->execute([$notificationId, $now]);
        return ['notification_id' => $notificationId, 'last_checked_at' => $now, 'last_event_id' => null];
    }
    return $row;
}

function update_checkpoint(PDO $pdo, int $notificationId, string $datetime, ?string $lastEventId = null): void {
    $pdo->prepare("UPDATE notification_checkpoints SET last_checked_at = ?, last_event_id = ? WHERE notification_id = ?")
        ->execute([$datetime, $lastEventId, $notificationId]);
}

function event_already_sent(PDO $pdo, int $notificationId, string $eventId): bool {
    $stmt = $pdo->prepare("SELECT 1 FROM notification_history WHERE notification_id = ? AND event_id = ? LIMIT 1");
    $stmt->execute([$notificationId, $eventId]);
    return (bool)$stmt->fetchColumn();
}

function record_history(PDO $pdo, array $data): void {
    $pdo->prepare("INSERT INTO notification_history (notification_id, client_id, account_login, trigger_type, event_id, event_data, message_sent, status, wa_message_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)")
        ->execute([
            $data['notification_id'],
            $data['client_id'],
            $data['account_login'],
            $data['trigger_type'],
            $data['event_id'],
            json_encode($data['event_data']),
            $data['message_sent'],
            $data['status'] ?? 'sent',
            $data['wa_message_id'] ?? null,
        ]);
}

function replace_placeholders(string $tpl, array $vars): string {
    foreach ($vars as $k => $v) {
        $tpl = str_replace('{{' . $k . '}}', (string)$v, $tpl);
    }
    return $tpl;
}

function lots_from_volume($v) {
    // MT5 often reports volume in minimal units; adjust if needed
    if (is_numeric($v)) {
        // heuristics: if large int, scale
        if ($v > 1000 && $v % 100 == 0) return $v / 100.0; // e.g., 100 = 1.00 lots
        if ($v > 100 && $v % 10 == 0) return $v / 100.0;
        return $v;
    }
    return $v;
}

while (true) {
    try {
        $notifications = get_enabled_notifications($pdo);
        $nowIso = date('Y-m-d H:i:s');

        foreach ($notifications as $n) {
            $checkpoint = get_checkpoint($pdo, (int)$n['id']);
            $fromTs = strtotime($checkpoint['last_checked_at'] ?? (date('Y-m-d H:i:s', time() - 60)));

            foreach ($n['account_logins'] as $login) {
                switch ($n['trigger_type']) {
                    case 'open_trade':
                        // Fetch recent deals since last check
                        $deals = $mt5->getDeals($login, $fromTs, null);
                        foreach ($deals as $deal) {
                            $eventId = (string)($deal['Deal'] ?? $deal['PositionID'] ?? $deal['Ticket'] ?? md5(json_encode($deal)));
                            if (event_already_sent($pdo, (int)$n['id'], $eventId)) continue;

                            // Determine if this is an open trade deal
                            $entry = $deal['Entry'] ?? $deal['Action'] ?? null; // 0 likely In
                            if ($entry !== null && (string)$entry !== '0') continue; // not an entry in

                            $symbol = $deal['Symbol'] ?? '';
                            $volume = lots_from_volume($deal['Volume'] ?? $deal['VolumeExt'] ?? 0);
                            $price  = $deal['Price'] ?? $deal['PriceOpen'] ?? '';

                            $msg = replace_placeholders($n['message'], [
                                'client_name' => $n['client_name'] ?? '',
                                'account_number' => $login,
                                'symbol' => $symbol,
                                'volume' => $volume,
                                'open_price' => $price,
                            ]);

                            $waId = null; $status = 'sent'; $err = null;
                            try {
                                if (!empty($n['client_phone'])) {
                                    $res = $wa->sendMessage($n['client_phone'], $msg);
                                    $waId = is_array($res) ? ($res['messages'][0]['id'] ?? null) : null;
                                } else {
                                    $status = 'failed';
                                    $err = 'Missing client phone';
                                }
                            } catch (Throwable $e) {
                                $status = 'failed';
                                $err = $e->getMessage();
                            }

                            record_history($pdo, [
                                'notification_id' => (int)$n['id'],
                                'client_id' => (string)$n['client_id'],
                                'account_login' => (int)$login,
                                'trigger_type' => $n['trigger_type'],
                                'event_id' => $eventId,
                                'event_data' => $deal,
                                'message_sent' => $msg,
                                'status' => $status,
                                'wa_message_id' => $waId,
                            ]);

                            echo '['.date('H:i:s')."] open_trade notif #{$n['id']} sent to {$n['client_phone']} (login {$login})\n";
                        }
                        break;
                    // Add other triggers here (close_trade, deposit, etc.)
                }
            }

            update_checkpoint($pdo, (int)$n['id'], $nowIso, null);
        }

        // Sleep between iterations
        sleep(10);
    } catch (Throwable $e) {
        fwrite(STDERR, 'Monitor error: ' . $e->getMessage() . "\n");
        sleep(5);
    }
}
