<?php
// api/notifications.php - CRUD operations for notification rules
header('Content-Type: application/json');
require_once __DIR__ . '/../config/bootstrap.php';

$method = $_SERVER['REQUEST_METHOD'];

function db_port() { return (int)($_ENV['DB_PORT'] ?? getenv('DB_PORT') ?: 3306); }
function db_host() { return $_ENV['DB_HOST'] ?? getenv('DB_HOST') ?: 'localhost'; }

// Fast connectivity check to avoid long PDO hang
$host = db_host(); $port = db_port();
$socket = @fsockopen($host, $port, $errno, $errstr, 2.0);
if (!$socket) {
    http_response_code(500);
    echo json_encode(['error' => 'Database is unreachable: '.$host.':'.$port.' ('.$errno.' '.$errstr.')']);
    exit;
}
@fclose($socket);

try {
    $pdo = get_pdo();
    if ($method === 'GET') {
        // List all notifications
        $stmt = $pdo->query("
            SELECT id, client_id, client_name, client_phone, account_logins, 
                   trigger_type, message, enabled, created_at, updated_at
            FROM notifications
            ORDER BY created_at DESC
        ");
        $notifications = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Format for frontend
        foreach ($notifications as &$n) {
            $n['enabled'] = (bool)$n['enabled'];
            $n['account_logins'] = json_decode($n['account_logins'], true) ?? [];
        }
        
        echo json_encode($notifications);
        
    } elseif ($method === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        $action = $input['action'] ?? '';
        
        if ($action === 'create') {
            // Create new notification rule
            $clients = $input['clients'] ?? [];
            $trigger = $input['trigger'] ?? '';
            $message = $input['message'] ?? '';
            
            if (empty($clients) || empty($trigger) || empty($message)) {
                throw new Exception('Missing required fields');
            }
            
            // Insert one notification per client
            $stmt = $pdo->prepare("
                INSERT INTO notifications 
                (client_id, client_name, client_phone, account_logins, trigger_type, message, enabled)
                VALUES (?, ?, ?, ?, ?, ?, 1)
            ");
            
            $count = 0;
            foreach ($clients as $client) {
                $clientId = $client['clientId'] ?? '';
                $name = $client['name'] ?? '';
                $phone = $client['phone'] ?? '';
                $logins = json_encode($client['logins'] ?? []);
                
                $stmt->execute([$clientId, $name, $phone, $logins, $trigger, $message]);
                $count++;
            }
            
            echo json_encode(['success' => true, 'count' => $count]);
            
        } elseif ($action === 'toggle') {
            // Toggle enabled/disabled
            $id = $input['id'] ?? 0;
            $enabled = $input['enabled'] ?? false;
            
            $stmt = $pdo->prepare("UPDATE notifications SET enabled = ? WHERE id = ?");
            $stmt->execute([$enabled ? 1 : 0, $id]);
            
            echo json_encode(['success' => true]);
            
        } elseif ($action === 'delete') {
            // Delete notification
            $id = $input['id'] ?? 0;
            
            $stmt = $pdo->prepare("DELETE FROM notifications WHERE id = ?");
            $stmt->execute([$id]);
            
            echo json_encode(['success' => true]);
            
        } else {
            throw new Exception('Unknown action');
        }
        
    } else {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
