# TradeNotify - Implementation Summary & Improvements

## Changes Completed

### 1. MT5 API Integration - Client-Based (✓ DONE)
- **Updated `src/MT5WebAPI.php`**:
  - Added `getClients($group)` - Fetches clients from `/api/client/get`
  - Added `getClientLogins($clientIds)` - Fetches account numbers from `/api/client/user/get_logins`
  - Added `getClientsWithAccounts($group)` - Combined method returning clients with their account numbers
  - Group filter: `skylinkscapital\A\Basic`

- **Updated `api/mt5_accounts.php`**:
  - Now returns client data instead of individual accounts
  - Response format:
    ```json
    [
      {
        "clientId": "1278",
        "name": "John Smith",
        "firstName": "John",
        "lastName": "Smith",
        "phone": "+1234567890",
        "email": "john@example.com",
        "logins": [18969, 18970, 18971],
        "loginCount": 3
      }
    ]
    ```

### 2. Required Files to Create/Update

#### A. `assets/admin.js` - Frontend Logic
**Key Features:**
- Load notifications tab by default (instead of triggers)
- Render saved notifications with enable/disable/delete options
- 3-step notification modal:
  - Step 1: Select clients (with back button from step 2/3)
  - Step 2: Select trigger (with back button)
  - Step 3: Compose message with visible placeholders (with back button)
- Dynamic placeholders based on trigger type
- Save button (instead of send) to persist to database

#### B. `api/notifications.php` - Backend API
**Endpoints:**
- `GET` - List all saved notifications
- `POST` with `action=create` - Save new notification rule
- `POST` with `action=toggle` - Enable/disable notification
- `POST` with `action=delete` - Delete notification

**Database Table:**
```sql
CREATE TABLE notifications (
    id INT AUTO_INCREMENT PRIMARY KEY,
    client_id VARCHAR(50) NOT NULL,
    client_name VARCHAR(255),
    client_phone VARCHAR(50),
    account_logins TEXT, -- JSON array of account numbers
    trigger_type VARCHAR(50) NOT NULL,
    message TEXT NOT NULL,
    enabled TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);
```

#### C. `public/index.php` - UI Updates
**Changes Needed:**
1. Update tab to show "Notifications" first (move to front)
2. Add back buttons to notification modal steps
3. Update Step 1 to show client list instead of account dropdown
4. Add placeholders display div in Step 3
5. Change "Send Notification" button to "Save"

**Step 1 HTML:**
```html
<div id="notification-step-1">
    <label class="block mb-2 font-semibold">Step 1: Select Client(s)</label>
    <div id="clients-list" class="max-h-64 overflow-y-auto space-y-2">
        <!-- Populated by JS -->
    </div>
    <button id="notification-next-1" class="mt-4 px-4 py-2 rounded bg-blue-500 text-white">Next</button>
</div>
```

**Step 2 HTML:**
```html
<div id="notification-step-2" class="hidden">
    <label class="block mb-2 font-semibold">Step 2: Select Trigger</label>
    <select id="notification-trigger" class="w-full mb-4 p-2 border rounded">
        <option value="open_trade">Start Trading (Open Trade)</option>
        <option value="close_trade">Close Trade</option>
        <option value="deposit">Deposit</option>
        <option value="withdrawal">Withdrawal</option>
        <option value="margin_call">Margin Call</option>
        <option value="stop_out">Stop Out</option>
    </select>
    <div class="flex gap-2">
        <button id="notification-back-2" class="px-4 py-2 rounded bg-gray-300 text-gray-700">Back</button>
        <button id="notification-next-2" class="px-4 py-2 rounded bg-blue-500 text-white">Next</button>
    </div>
</div>
```

**Step 3 HTML:**
```html
<div id="notification-step-3" class="hidden">
    <label class="block mb-2 font-semibold">Step 3: Compose Message</label>
    <div id="placeholders-list" class="mb-3">
        <!-- Populated by JS based on trigger -->
    </div>
    <textarea id="notification-message" class="w-full mb-4 p-2 border rounded" rows="4" 
        placeholder="Type your message. Click placeholders above to insert."></textarea>
    <div class="flex gap-2">
        <button id="notification-back-3" class="px-4 py-2 rounded bg-gray-300 text-gray-700">Back</button>
        <button id="notification-save" class="px-4 py-2 rounded bg-green-500 text-white">Save Notification Rule</button>
    </div>
</div>
```

### 3. Trigger Monitoring System

#### D. `cli/monitor_triggers.php` - Background Service
**Purpose:** Continuously monitor MT5 events and send WhatsApp messages based on saved notification rules

**Architecture:**
```php
<?php
// Run continuously: php cli/monitor_triggers.php
require_once __DIR__ . '/../config/bootstrap.php';
require_once __DIR__ . '/../src/MT5WebAPI.php';
require_once __DIR__ . '/../src/WhatsAppClient.php';

while (true) {
    // 1. Load all enabled notifications from database
    $notifications = loadEnabledNotifications();
    
    // 2. For each notification, check MT5 for trigger events
    foreach ($notifications as $notification) {
        switch ($notification['trigger_type']) {
            case 'open_trade':
                checkOpenTrades($notification);
                break;
            case 'close_trade':
                checkClosedTrades($notification);
                break;
            case 'deposit':
                checkDeposits($notification);
                break;
            // ... other triggers
        }
    }
    
    sleep(10); // Check every 10 seconds
}
```

**Event Detection:**
- Store last checked timestamp for each notification
- Query MT5 deals/positions since last check
- Match events to notification rules
- Send WhatsApp message with placeholder replacements
- Log sent messages to prevent duplicates

### 4. Placeholder System

**Available Placeholders by Trigger:**

| Trigger Type | Placeholders |
|---|---|
| `open_trade` | `{{client_name}}`, `{{account_number}}`, `{{symbol}}`, `{{volume}}`, `{{open_price}}` |
| `close_trade` | `{{client_name}}`, `{{account_number}}`, `{{symbol}}`, `{{volume}}`, `{{close_price}}`, `{{profit}}` |
| `deposit` | `{{client_name}}`, `{{account_number}}`, `{{amount}}`, `{{balance}}` |
| `withdrawal` | `{{client_name}}`, `{{account_number}}`, `{{amount}}`, `{{balance}}` |
| `margin_call` | `{{client_name}}`, `{{account_number}}`, `{{margin_level}}`, `{{equity}}` |
| `stop_out` | `{{client_name}}`, `{{account_number}}`, `{{margin_level}}`, `{{equity}}` |

**Example Message:**
```
Hello {{client_name}}! 

Your account {{account_number}} just opened a {{symbol}} trade with {{volume}} lots at {{open_price}}.

Good luck!
```

### 5. Database Migration

**Run this SQL:**
```sql
-- Add notifications table
CREATE TABLE IF NOT EXISTS notifications (
    id INT AUTO_INCREMENT PRIMARY KEY,
    client_id VARCHAR(50) NOT NULL COMMENT 'MT5 Client RecordID',
    client_name VARCHAR(255),
    client_phone VARCHAR(50),
    account_logins TEXT COMMENT 'JSON array of account login numbers',
    trigger_type VARCHAR(50) NOT NULL COMMENT 'open_trade, close_trade, deposit, etc.',
    message TEXT NOT NULL COMMENT 'WhatsApp message template with placeholders',
    enabled TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_enabled (enabled),
    INDEX idx_trigger (trigger_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Add table to track sent notifications (prevent duplicates)
CREATE TABLE IF NOT EXISTS notification_history (
    id INT AUTO_INCREMENT PRIMARY KEY,
    notification_id INT NOT NULL,
    client_id VARCHAR(50) NOT NULL,
    account_login INT NOT NULL,
    trigger_type VARCHAR(50) NOT NULL,
    event_id VARCHAR(255) COMMENT 'MT5 deal/position ID',
    message_sent TEXT,
    sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    wa_message_id VARCHAR(255),
    FOREIGN KEY (notification_id) REFERENCES notifications(id) ON DELETE CASCADE,
    UNIQUE KEY unique_event (notification_id, event_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
```

## Suggested Improvements

### 1. **Batch Notifications**
- Allow sending test message to all selected clients immediately
- Useful for announcements or urgent alerts

### 2. **Notification Templates**
- Save common message templates
- Quick select from template library
- Pre-filled placeholders

### 3. **Client Groups**
- Create custom client groups (VIP, Standard, Demo)
- Apply notifications to entire groups
- Easier management

### 4. **Scheduling**
- Time-based triggers (e.g., daily balance report)
- Quiet hours (don't send between 10 PM - 8 AM)
- Timezone support per client

### 5. **Advanced Filters**
- Trigger conditions (e.g., only send if profit > $100)
- Account balance thresholds
- Symbol filters (only EUR/USD trades)

### 6. **Analytics Dashboard**
- Messages sent per day/week/month
- Client engagement metrics
- Most triggered events

### 7. **Multi-Language Support**
- Detect client language from MT5
- Auto-translate messages
- Language-specific templates

### 8. **Rate Limiting**
- Prevent spam (max X messages per client per hour)
- WhatsApp API quota management
- Queue system for high volume

### 9. **Rich Messages**
- WhatsApp templates with buttons
- Images/charts (account performance)
- Interactive menus

### 10. **Client Portal**
- Clients can manage their own notification preferences
- Opt-in/opt-out specific triggers
- Update phone number

## Next Steps

1. Create database migration (run SQL above)
2. Create `api/notifications.php` with CRUD operations
3. Update `public/index.php` with new modal HTML
4. Create `assets/admin.js` with all logic
5. Create `cli/monitor_triggers.php` for background monitoring
6. Test workflow end-to-end
7. Deploy monitor as systemd service or Windows Task

## Testing Checklist

- [ ] Clients load from MT5 API correctly
- [ ] Can select multiple clients
- [ ] Back buttons work in modal
- [ ] Placeholders display based on trigger
- [ ] Save creates database record
- [ ] Saved notifications appear in list
- [ ] Toggle enable/disable works
- [ ] Delete removes notification
- [ ] Monitor detects MT5 events
- [ ] WhatsApp messages sent with correct placeholders
- [ ] No duplicate messages sent

## Files Modified Summary

1. ✅ `src/MT5WebAPI.php` - Added client methods
2. ✅ `api/mt5_accounts.php` - Returns clients instead of accounts
3. ✅ `.env` - Fixed password quoting (# character issue)
4. ⏳ `api/notifications.php` - NEW FILE (CRUD for notifications)
5. ⏳ `assets/admin.js` - NEEDS RECREATION (corrupted, need full rewrite)
6. ⏳ `public/index.php` - UPDATE (modal HTML needs back buttons + client list)
7. ⏳ `cli/monitor_triggers.php` - NEW FILE (background service)
8. ⏳ `database/migrations/` - NEW SQL (notifications tables)

